/**************************************************************************************

   Copyright (c) Hilscher GmbH. All Rights Reserved.

 **************************************************************************************

   Filename:
    $Workfile: $
   Last Modification:
    $Author: stephans $
    $Modtime: $
    $Revision: 6507 $

   Targets:
     MCF51CN128          : yes

   Description:
    Coldfire MCF51CN128 OS Abstraction Implementation

   Changes:

     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     1         01.03.2011  SS       initial version

**************************************************************************************/

/* Standard includes */
#include <string.h>
#include <ctype.h>

/* Coldfire includes */
#include <hidef.h>
#include <UART.h>
#include "mcf51cn128.h"

/* Toolkit includes */
#include "cifXToolkit.h"
#include "OS_Dependent.h"
#include "cifXErrors.h"

/* Serial DPM includes */
#include "OS_Spi.h"

static volatile uint32_t s_ulTime      = 0;
static          int      s_fOSInitDone = 0;

#if (CPU_FREQ_50MHZ == 1) & (CPU_FREQ_37MHZ == 1)
#error "Only one CPU frequency must be selected"
#elif (CPU_FREQ_50MHZ == 0) & (CPU_FREQ_37MHZ == 0)
#error "One CPU frequency must be selected"
#endif

#if (NX50_PARDPM8 == 0) & (NX50_PARDPM16 == 0) & (NX50_SERDPM == 0)
#error "One interface must be selected"
#endif

/*****************************************************************************/
/*! Interrupt Routine for millisecond timer generation                       */
/*****************************************************************************/
static __interrupt VectorNumber_Vtpm1ovf void isrVmTim1(void)
{
  s_ulTime++;
  TPM1SC_TOF;
  TPM1SC_TOF = 0;
}

/*****************************************************************************/
/*! Init Uart                                                                */
/*****************************************************************************/
UARTError InitializeUART(UARTBaudRate baudRate)
{
  UNREFERENCED_PARAMETER(baudRate);
  
  PTDPF2_D0 = 0b10;
  PTDPF2_D1 = 0b10;
  
  SCI1C1_M   = 0b0; /* start + 8 data bits (lsb first) + stop */
  SCI1C1_PE  = 0b0; /* No hardware parity generation or checking */
  SCI1C1_ILT = 0b0; /* Idle character bit count starts after start bit */
  
  SCI1C2_TE  = 0b1; /* Transmitter on */
  SCI1C2_RE  = 0b1; /* Receiver on */
  SCI1C2_RIE = 0b0; /* Receiver Interrupt Disable */

  #if (CPU_FREQ_50MHZ == 1)  
    SCI1BD_SBR = 27; /* Set Baudrate to 115.2 kbps @ 50MHz cpu clock */
  #elif (CPU_FREQ_37MHZ == 1)
    SCI1BD_SBR = 20; /* Set Baudrate to 115.2 kbps @ 37MHz cpu clock */
  #endif
  return kUARTNoError; /* Return No Error */
}

/*****************************************************************************/
/*! Write the specified amount of data to the SCI file specified by
 *  handle from the given buffer.                                            */
/*****************************************************************************/
UARTError WriteUARTN(const void* bytes, unsigned long length)
{
  char *bData =(char *)bytes;
  
  while (length--)
  {
    while (!SCI1S1_TC)
      ;  
    SCI1D = *bData++;
  }

  return kUARTNoError; /* Return No Error */
}

/*****************************************************************************/
/*! Read the specified amount of data to the SCI file specified by
 *  handle from the given buffer.                                            */
/*****************************************************************************/
UARTError ReadUARTN(void* bytes, unsigned long length)
{
  char *bData =(char *)bytes;
  
  while (length--)
  {
    while (!SCI1S1_RDRF)
      ;  
    *bData = SCI1D;
    
    WriteUARTN(bData, 1);
    bData++;
  }

  return kUARTNoError; /* Return No Error */
}

/*****************************************************************************/
/*! Init 25Mhz crystal oscillator                                            */
/*****************************************************************************/
static void clock_init_25MHz() 
{
   /* 1. First, FEI must transition to FBE mode:
    * a) MCGC2 = 0x36 (%00110110)
    *  BDIV (bits 7 and 6) set to %00, or divide-by-1
    *  RANGE (bit 5) set to 1 because the frequency of 8 MHz is within the high frequency range
    *  HGO (bit 4) set to 1 to configure the crystal oscillator for high gain operation
    *  EREFS (bit 2) set to 1, because a crystal is being used
    *  ERCLKEN (bit 1) set to 1 to ensure the external reference clock is active */
   MCGC2_BDIV    = 0;
   MCGC2_RANGE   = 1;
   MCGC2_HGO     = 1;
   MCGC2_EREFS   = 1;
   MCGC2_ERCLKEN = 1;   

   /* b) Loop until OSCINIT (bit 1) in MCGSC is 1, indicating the crystal selected 
    *    by the EREFS bit has been initialized. */
   while (!(MCGSC&MCGSC_OSCINIT_MASK))
     ;

   /* c) Because RANGE = 1, set DIV32 (bit 4) in MCGC3 to allow access to the proper
    *    RDIV bits while in an FLL external mode. */
   MCGC3_DIV32 = 1;
   
   /* d) MCGC1 = 0x98 (%10011000)
    *  CLKS (bits 7 and 6) set to %10 to select external reference clock as system clock source
    *  RDIV (bits 5-3) set to %100, or divide-by-16 because 25MHz / 16 = 1.5625 kHz that 
    *   is in the 1 MHz to 2 MHz range required by the PLL
    *  IREFS (bit 2) cleared to 0, selecting the external reference clock */
   MCGC1_CLKS  = 0b10; // CLKS = 10 -> external reference clock
   MCGC1_RDIV  = 0b100; // RDIV = 2^4 -> 25MHz/16 = 1.5625 MHz for PLL
   MCGC1_IREFS = 0;
   
   /* wait for Reference Status bit to update */ 
   while (!(MCGSC&MCGSC_IREFST_MASK))
     ;
   
   /* and for clock status bits to update */
   while (MCGSC_CLKST != 0b10)
     ; 

   /* switch from FBE to PBE (PLL bypassed internal) mode 
    * set PLL multi 50MHz amd select PLL */      
   MCGC3_PLLS = 1; 

   #if (CPU_FREQ_50MHZ == 1)
     MCGC3_VDIV = 0b1000; /* Multiply by 32 */
   #elif (CPU_FREQ_37MHZ == 1)
     MCGC3_VDIV = 0b0110; /* Multiply by 24 */
   #endif
     
   /* wait for PLL status bit and LOCK bit */
   while (!(MCGSC&MCGSC_PLLST_MASK) && !(MCGSC&MCGSC_LOCK_MASK))
     ;

   /* switch from PBE to PEE (PLL enabled external mode) */                                    
   MCGC1_CLKS = 0b11;

   /* Wait for clock status bits to update */
   while (MCGSC_CLKST != 0b11)
     ; 

   /* PEE mode entered */
}

/*****************************************************************************/
/*! Init 1kHz timer                                                          */
/*****************************************************************************/
static void timer_init_1kHz() 
{
  /* Init millisecond timer for OS_GetMiliSecCounter() */
  TPM1SC_CLKSx = 0b00;   /* Stop TPM counter */
  TPM1SC_PS    = 0b000;  /* Set divisor to 1 making the clock 40ns*/
  #if (CPU_FREQ_50MHZ == 1)  
    TPM1MOD      = 0x61A8; /* Set Modulo for a count of 25000 =~ 1ms */
  #elif (CPU_FREQ_37MHZ == 1)
    TPM1MOD      = 0x4844; /* Set Modulo for a count of 18500 =~ 1ms */
  #endif
  TPM1CNT      = 0x0000; /* Clear the counter */
  TPM1SC_TOIE  = 0b1;    /* Set Interrupt Enable */
  TPM1SC_CLKSx = 0b01;   /* Start TPM counter with clock source BUSCLK */
}

/*****************************************************************************/
/*! Init FlexBus                                                             */
/*****************************************************************************/
static void flexbus_init() 
{
  /* J0 - J5 */
  PTJPF2_J0 = 0b01;
  PTJPF2_J1 = 0b10;
  PTJPF2_J2 = 0b10;
  PTJPF2_J3 = 0b10;
  PTJPF1_J4 = 0b10;
  PTJPF1_J5 = 0b10;

  /* G0 - G7 */ 
  PTGPF2_G0 = 0b10;
  PTGPF2_G1 = 0b10;
  PTGPF2_G2 = 0b10;
  PTGPF2_G3 = 0b10;
  PTGPF1_G4 = 0b10;
  PTGPF1_G5 = 0b10;
  PTGPF1_G6 = 0b10;
  PTGPF1_G7 = 0b10;

  /* F0 - F7 */ 
  PTFPF2_F0 = 0b10;
  PTFPF2_F1 = 0b10;
  PTFPF2_F2 = 0b10;
  PTFPF2_F3 = 0b10;
  PTFPF1_F4 = 0b10;
  PTFPF1_F5 = 0b10;
  PTFPF1_F6 = 0b10;
  PTFPF1_F7 = 0b10;

  /* H0 - H7 */ 
  PTHPF2_H0 = 0b10;
  PTHPF2_H1 = 0b10;
  PTHPF2_H2 = 0b10;
  PTHPF2_H3 = 0b10;
  PTHPF1_H4 = 0b10;
  PTHPF1_H5 = 0b10;
  PTHPF1_H6 = 0b10;
  PTHPF1_H7 = 0b10;

  /* E7 - E6 */ 
  PTEPF1_E6 = 0b10;
  PTEPF1_E7 = 0b10;
 
  /* C4 */ 
  PTCPF1_C4 = 0b01;

  /* Chip-Select Mask Registers */
  CSMR0_V  = 0b1; 
  
  /* Chip-Select Address Register @ 0x00400000 */
  CSAR0_BA = 0x40;
  
  /* Chip-Select Control Registers */
  #if (NX50_PARDPM8 == 1) 
    CSCR0_PS   = 0b01; /* 8-bit port size */
    CSCR0_MUX  = 0b1;  /* multiplexed configuration */
  #elif (NX50_PARDPM16 == 1)
    CSCR0_PS   = 0b10; /* 16-bit port size */
    CSCR0_MUX  = 0b1;  /* multiplexed configuration */
  #endif
  CSCR0_AA   = 0b1;  /* Auto-acknowledge enable */
  CSCR0_WS   = 0x5;  /* Wait states */
  CSCR0_ASET = 0b01; /* Assert FB_CSn on second rising clock 
                        edge after address is asserted */
  CSCR0_WRAH = 0;
  CSCR0_RDAH = 0;
}

/*****************************************************************************/
/*! Init SPI                                                                 */
/*****************************************************************************/
static void init_spi(void)
{
  /* Set pin function for port C pins 4,5,6, and 7*/
  PTCPF1_C4 = 0b00;
  PTCPF1_C5 = 0b10;
  PTCPF1_C6 = 0b10;
  PTCPF1_C7 = 0b10;
  
  /* Setup CS as GPIO Toggle Manually PTC4 */
  PTCDD_PTCDD4 = 1;
  
  /* Set CS to high to start Code to toggle low when needed */ 
  PTCD_PTCD4 = 1;  

  /* SPI Baud Rate Register */
  SPI1BR_SPR  = 0b000; /* SPI Baud Rate Divisor = 2 */ 
  #if (CPU_FREQ_50MHZ == 1)  
    SPI1BR_SPPR = 0b001; /* Prescaler Divisor = 2  => SPI Clk: 6.125MHz */
  #elif (CPU_FREQ_37MHZ == 1)
    SPI1BR_SPPR = 0b000; /* Prescaler Divisor = 1 => SPI Clk: 9.37MHz */
  #endif
  
  /* SPI Control Register */
  SPI1C1_CPHA = 0b1;
  SPI1C1_CPOL = 0b1;
  SPI1C1_MSTR = 0b1;
  SPI1C1_SPE  = 0b1;
  SPI1C2      = 0x00;
}

/*****************************************************************************/
/*! Assert chip select
*   \param pvOSDependent OS Dependent parameter                              */
/*****************************************************************************/
void OS_SpiAssert(void* pvOSDependent)
{
  UNREFERENCED_PARAMETER(pvOSDependent);
  PTCD_PTCD4 = 0;
}

/*****************************************************************************/
/*! Deassert chip select
*   \param pvOSDependent OS Dependent parameter                              */
/*****************************************************************************/
void OS_SpiDeassert(void* pvOSDependent)
{
  UNREFERENCED_PARAMETER(pvOSDependent);
  PTCD_PTCD4 = 1;
}

/*****************************************************************************/
/*! Transfer byte stream via SPI. As send and receive buffer are optional,
*   the routine must be capable of sending dummy bytes (in case pbSend == NULL)
*   and discard receive bytes (if pbRecv == NULL). If caller does not pass
*   any buffer at all, we are dealing with an idle transfer to waste some time.
*   \param pvOSDependent OS Dependent parameter
*   \param pbSend        Send buffer (Can be NULL for polling data from slave)
*   \param pbRecv        Receive buffer (Can be NULL if slaves received data
                         is discarded by caller)
*   \param ulLen         Length of SPI transfer                              */
/*****************************************************************************/
void OS_SpiTransfer(void* pvOSDependent, uint8_t* pbSend, uint8_t* pbRecv, uint32_t ulLen)
{
  UNREFERENCED_PARAMETER(pvOSDependent);

  while (ulLen--)
  {
    /* Read SPI status reg before sending byte */
    while(!(SPI1S&SPI1S_SPTEF_MASK)) {};

    /* write byte to spi bus */
    if (pbSend == NULL)
      SPI1D = 0x00;
    else
      SPI1D = *(pbSend++);

    /* Read SPI status reg before reading byte */
    while(!(SPI1S&SPI1S_SPRF_MASK)) {};

    /* get the received byte */
    if (pbRecv == NULL)
      SPI1D;
    else
      *(pbRecv++) = SPI1D;
  }
}

/*****************************************************************************/
/*! O/S Specific initialization (initializes libpciaccess)
*     \return CIFX_NO_ERROR on success                                       */
/*****************************************************************************/
long OS_Init(void)
{
  /* Derive system clock from external 25MHz crystal */
  clock_init_25MHz();
  
  /* Init millisecond timer for OS_GetMiliSecCounter() */
  timer_init_1kHz(); 

  #if (NX50_PARDPM8 == 1) | (NX50_PARDPM16 == 1)
    /* Init flexbus */
    flexbus_init();
  #elif (NX50_SERDPM == 1)  
    /* Init SPI */
    init_spi();
  #endif
     
  /* Disable Watchdog */
  SOPT1_COPT = 0b00;
  
  EnableInterrupts;
  
  s_fOSInitDone = 1;
  
  return CIFX_NO_ERROR;
}

/*****************************************************************************/
/*! O/S Specific de-initialization (de-initializes libpciaccess)             */
/*****************************************************************************/
void OS_Deinit(void)
{
}

/*****************************************************************************/
/*! Memory allocation wrapper (standard malloc)
*     \param ulSize Size of block to allocate
*     \return NULL on failure                                                */
/*****************************************************************************/
void* OS_Memalloc(uint32_t ulSize)
{
  return malloc(ulSize);
}

/*****************************************************************************/
/*! Memory de-allocation wrapper (standard free)
*     \param pvMem  Block to free                                            */
/*****************************************************************************/
void OS_Memfree(void* pvMem)
{
  free(pvMem);
}

/*****************************************************************************/
/*! Memory resize wrapper (standard realloc)
*     \param pvMem      Block to resize
*     \param ulNewSize  New size of the block
*     \return NULL on error                                                  */
/*****************************************************************************/
void* OS_Memrealloc(void* pvMem, uint32_t ulNewSize)
{
  return realloc(pvMem, ulNewSize);
}

/*****************************************************************************/
/*! Memset wrapper
*     \param pvMem   Memory to set
*     \param bFill   Fill byte
*     \param ulSize  Size of the fill block                                  */
/*****************************************************************************/
void OS_Memset(void* pvMem, uint8_t bFill, uint32_t ulSize)
{
  memset(pvMem, bFill, ulSize);
}

/*****************************************************************************/
/*! Memcopy wrapper
*     \param pvDest  Destination pointer
*     \param pvSrc   Source pointer
*     \param ulSize  Size to copy                                            */
/*****************************************************************************/
void OS_Memcpy(void* pvDest, void* pvSrc, uint32_t ulSize)
{
  memcpy(pvDest, pvSrc, ulSize);
}

/*****************************************************************************/
/*! Memcompare wrapper
*     \param pvBuf1  First compare buffer
*     \param pvBuf2  Second compare buffer
*     \param ulSize  Size to compare
*     \return 0 if blocks are equal                                          */
/*****************************************************************************/
int OS_Memcmp(void* pvBuf1, void* pvBuf2, uint32_t ulSize)
{
  return memcmp(pvBuf1, pvBuf2, ulSize);
}

/*****************************************************************************/
/*! Memmove wrapper (Overlapping memory copy)
*     \param pvDest  Destination buffer
*     \param pvSrc   Source buffer
*     \param ulSize  Size to move                                            */
/*****************************************************************************/
void OS_Memmove(void* pvDest, void* pvSrc, uint32_t ulSize)
{
  memmove(pvDest, pvSrc, ulSize);
}

/*****************************************************************************/
/*! Read PCI configuration area of specified card
*     \param pvOSDependent OS Dependent parameter to identify card
*     \return Pointer to configuration data (passed to WritePCIConfig)       */
/*****************************************************************************/
void* OS_ReadPCIConfig(void* pvOSDependent)
{
  UNREFERENCED_PARAMETER(pvOSDependent);
  return NULL;
}

/*****************************************************************************/
/*! Restore PCI configuration
*     \param pvOSDependent OS Dependent parameter to identify card
*     \param pvPCIConfig   Pointer returned from ReadPCIConfig               */
/*****************************************************************************/
void OS_WritePCIConfig(void* pvOSDependent, void* pvPCIConfig)
{
  UNREFERENCED_PARAMETER(pvOSDependent);
  UNREFERENCED_PARAMETER(pvPCIConfig);
}

static void* s_pvIsrDevInst = NULL;
/*****************************************************************************/
/*! Interrupt Routine for DPM interrupt                                      */
/*****************************************************************************/
static __interrupt VectorNumber_Vkeyboard void isrVkeyboard(void)
{
  PDEVICEINSTANCE ptDevInst = (PDEVICEINSTANCE)s_pvIsrDevInst;
  int iISRRet              = CIFX_TKIT_IRQ_OTHERDEVICE;

  iISRRet = cifXTKitISRHandler(ptDevInst, 1);
 
  switch(iISRRet)
  {
    case CIFX_TKIT_IRQ_DSR_REQUESTED:
      cifXTKitDSRHandler(ptDevInst);
      break;
    
    case CIFX_TKIT_IRQ_HANDLED:
      /* Everything was done by ISR, no need to call DSR */
      break;
    
    case CIFX_TKIT_IRQ_OTHERDEVICE:
    default:
      /* This is an interrupt from another device. pciIntConnect creates a linked 
         list of ISRs that are sharing the same interrupt line. When the interrupt 
         occurs, it sequentially invokes all the ISRs connected to it.  */
      break;
  }
  
  KBI2SC_KBACK = 0b1;
}

/*****************************************************************************/
/*! Enable interrupts on the given device
*     \param pvOSDependent Pointer to internal device structure              */
/*****************************************************************************/
void OS_EnableInterrupts(void* pvOSDependent)
{
  if (NULL == pvOSDependent)
    return;
  
  s_pvIsrDevInst = pvOSDependent;
  
  /* Enable Interrupt on PTE5/KBI2P5 */
  PTEPF1_E5     = 0b01;/* KBI2P5 */
  KBI2SC_KBIE   = 0b0; /* Mask interrupts by clearing  */
  KBI2ES_KBEDG5 = 0b1; /* detects rising edge/high level for interrupt generation */
  PTEPE_PTEPE5  = 0b0; /* Internal pull-up device disabled */
  KBI2PE_KBIPE5 = 0b1; /* Pin enabled as interrupt */
  KBI2SC_KBIMOD = 0b1; /* pins detect edges and levels */
  KBI2SC_KBACK  = 0b1; /* clear any false interrupts */
  KBI2SC_KBIE   = 0b1; /* KBI Interrupt enable */  
}

/*****************************************************************************/
/*! Disable interrupts on the given device
*     \param pvOSDependent Pointer to internal device structure              */
/*****************************************************************************/
void OS_DisableInterrupts(void* pvOSDependent)
{
  UNREFERENCED_PARAMETER(pvOSDependent);
  
  s_pvIsrDevInst = NULL;
  
  /* Disable Interrupt on PTE5/KBI2P5 */
  KBI2SC_KBIE   = 0b0; /* Mask interrupts by clearing  */
  KBI2PE_KBIPE5 = 0b0; /* Pin enabled as interrupt */
}

/*****************************************************************************/
/*! Open file for reading
*     \param szFilename   File to open (including path)
*     \param pulFileSize  Returned size of the file in bytes
*     \return Handle to the file, NULL on failure                            */
/*****************************************************************************/
void* OS_FileOpen(char* szFile, uint32_t* pulFileLen)
{
  UNREFERENCED_PARAMETER(szFile);
  UNREFERENCED_PARAMETER(pulFileLen);
  return NULL;
}

/*****************************************************************************/
/*! Read data from file
*     \param pvFile    Handle to the file (acquired by OS_FileOpen)
*     \param ulOffset  Offset to read from
*     \param ulSize    Size to read
*     \param pvBuffer  Buffer to read data into
*     \return number of bytes read                                           */
/*****************************************************************************/
uint32_t OS_FileRead(void* pvFile, uint32_t ulOffset, uint32_t ulSize, void* pvBuffer)
{
  UNREFERENCED_PARAMETER(pvFile);
  UNREFERENCED_PARAMETER(ulOffset);  
  UNREFERENCED_PARAMETER(ulSize);
  UNREFERENCED_PARAMETER(pvBuffer);
  return 0;
}

/*****************************************************************************/
/*! Close open file
*     \param pvFile    Handle to the file (acquired by OS_FileOpen)          */
/*****************************************************************************/
void OS_FileClose(void* pvFile)
{
  UNREFERENCED_PARAMETER(pvFile);
}

/*****************************************************************************/
/*! Get Millisecond counter value (used for timeout handling)
*     \return Counter value with a resolution of 1ms                         */
/*****************************************************************************/
uint32_t OS_GetMilliSecCounter(void)
{
  return s_ulTime;
}

/*****************************************************************************/
/*! Sleep for the given time
*     \param ulSleepTimeMs Time in ms to sleep (0 will sleep for 50us)       */
/*****************************************************************************/
void OS_Sleep(uint32_t ulSleepTimeMs)
{
  uint32_t ulStart = 0;
  uint32_t ulStop  = 0;
  
  if (!s_fOSInitDone)
    return;
  
  ulStart = OS_GetMilliSecCounter();
  do
  {
    ulStop = OS_GetMilliSecCounter();
  } while (ulStop - ulStart < ulSleepTimeMs);
}

/*****************************************************************************/
/*! Create Lock (Usually same as mutex, but does not support timed waiting)
*     \return Handle to created lock                                         */
/*****************************************************************************/
void* OS_CreateLock(void)
{
  return OS_CreateMutex();
}

/*****************************************************************************/
/*! Acquire a lock
*     \param pvLock Handle to lock                                           */
/*****************************************************************************/
void OS_EnterLock(void* pvLock)
{
  OS_WaitMutex(pvLock, 0);
}

/*****************************************************************************/
/*! Release a lock
*     \param pvLock Handle to lock                                           */
/*****************************************************************************/
void OS_LeaveLock(void* pvLock)
{
  OS_ReleaseMutex(pvLock);
}

/*****************************************************************************/
/*! Delete a lock
*     \param pvLock Handle to lock                                           */
/*****************************************************************************/
void OS_DeleteLock(void* pvLock)
{
  OS_DeleteMutex(pvLock);
}

/*****************************************************************************/
/*! Create mutex
*     \return Handle to new created mutex                                    */
/*****************************************************************************/
void* OS_CreateMutex(void)
{
  return (void*)1; 
}

/*****************************************************************************/
/*! Try to acquire mutex with timeout
*     \param pvMutex   Handle to mutex
*     \param ulTimeout Timeout in ms to wait for mutex
*     \return !=0 if mutex was acquired                                      */
/*****************************************************************************/
int OS_WaitMutex(void* pvMutex, uint32_t ulTimeout)
{ 
  UNREFERENCED_PARAMETER(pvMutex);
  UNREFERENCED_PARAMETER(ulTimeout);
  
  return 1;  
}

/*****************************************************************************/
/*! Release previously acquired mutex
*     \param pvMutex   Handle to mutex                                       */
/*****************************************************************************/
void OS_ReleaseMutex(void* pvMutex)
{
  UNREFERENCED_PARAMETER(pvMutex);
}

/*****************************************************************************/
/*! Delete mutex
*     \param pvMutex   Handle to mutex                                       */
/*****************************************************************************/
void OS_DeleteMutex(void* pvMutex)
{
  UNREFERENCED_PARAMETER(pvMutex);
}

/*****************************************************************************/
/*! Create event
*     \return Handle to created event                                        */
/*****************************************************************************/
void* OS_CreateEvent(void)
{
  return (void*)1; 
}

/*****************************************************************************/
/*! Signal event
*     \param pvEvent Handle to event                                         */
/*****************************************************************************/
void OS_SetEvent(void* pvEvent)
{
  UNREFERENCED_PARAMETER(pvEvent);
}

/*****************************************************************************/
/*! Reset event
*     \param pvEvent Handle to event                                         */
/*****************************************************************************/
void OS_ResetEvent(void* pvEvent)
{
  UNREFERENCED_PARAMETER(pvEvent);
}

/*****************************************************************************/
/*! Delete event
*     \param pvEvent Handle to event                                         */
/*****************************************************************************/
void OS_DeleteEvent(void* pvEvent)
{
  UNREFERENCED_PARAMETER(pvEvent);
}

/*****************************************************************************/
/*! Wait for event
*     \param pvEvent   Handle to event
*     \param ulTimeout Timeout in ms to wait for event
*     \return CIFX_EVENT_SIGNALLED if event was set, CIFX_EVENT_TIMEOUT otherwise */
/*****************************************************************************/
uint32_t OS_WaitEvent(void* pvEvent, uint32_t ulTimeout)
{
  UNREFERENCED_PARAMETER(pvEvent);
  UNREFERENCED_PARAMETER(ulTimeout);

  return CIFX_EVENT_SIGNALLED; 
}

/*****************************************************************************/
/*! Map driver pointer to user space
*     \param pvDriverMem  Pointer to driver memory
*     \param ulMemSize    Size of the memory to map
*     \param ppvMappedMem Returned mapped pointer
*     \param os_dependent OS Dependent parameter in DEVICEINSTANCE
*     \return Handle to mapping, NULL on error                               */
/*****************************************************************************/
void* OS_MapUserPointer(void* pvDriverMem, uint32_t ulMemSize, void** ppvMappedMem, void* pvOSDependent)
{
  UNREFERENCED_PARAMETER(ulMemSize);
  UNREFERENCED_PARAMETER(pvOSDependent);
  /* We are running in user mode, so it is not necessary to map anything to user space */ 
  *ppvMappedMem = pvDriverMem;
  
  return pvDriverMem;
}

/*****************************************************************************/
/*! Unmap previously mapped user space pointer 
*     \param phMapping  Handle returned from OS_MapUserPointer
*     \param os_dependent OS Dependent parameter in DEVICEINSTANCE
*     \return 0 on error                                                     */
/*****************************************************************************/
int OS_UnmapUserPointer(void* phMapping, void* pvOSDependent)
{
  UNREFERENCED_PARAMETER(phMapping);
  UNREFERENCED_PARAMETER(pvOSDependent);
  /* We are running in user mode, so it is not necessary to map anything to user space */ 
  return 1;
}

/*****************************************************************************/
/*! Compare strings
*     \param pszBuf1  String buffer 1
*     \param pszBuf2  String buffer 2
*     \return 0 if strings are equal                                         */
/*****************************************************************************/
int OS_Strcmp(const char* pszBuf1, const char* pszBuf2)
{

  return strcmp(pszBuf1, pszBuf2);
}

/*****************************************************************************/
/*! Compare strings case insensitive
*     \param pszBuf1  String buffer 1
*     \param pszBuf2  String buffer 2
*     \param ulLen    Maximum length to compare
*     \return 0 if strings are equal                                         */
/*****************************************************************************/
int OS_Strnicmp(const char* pszBuf1, const char* pszBuf2, uint32_t ulLen)
{
  char* szBuff1Copy = malloc(strlen(pszBuf1) + 1);
  char* szBuff2Copy = malloc(strlen(pszBuf2) + 1);
  int   iRet;
  int   iTmp;

  for(iTmp = 0; iTmp < min(strlen(pszBuf1), ulLen); ++iTmp)
    szBuff1Copy[iTmp] = (char)tolower((int)pszBuf1[iTmp]);

  for(iTmp = 0; iTmp < min(strlen(pszBuf2), ulLen); ++iTmp)
    szBuff2Copy[iTmp] = (char)tolower((int)pszBuf2[iTmp]);

  iRet = strncmp(szBuff1Copy, szBuff2Copy, ulLen);

  free(szBuff1Copy);
  free(szBuff2Copy);

  return iRet;
}

/*****************************************************************************/
/*! Get length of string
*     \param szText  Text buffer
*     \return Length of given string                                         */
/*****************************************************************************/
int OS_Strlen(const char* szText)
{
  return (int)strlen(szText);
}

/*****************************************************************************/
/*! Copy string to destination buffer
*     \param szText   Destination string
*     \param szSource Source string
*     \param ulLen    Maximum length to copy
*     \return Pointer to szDest                                              */
/*****************************************************************************/
char* OS_Strncpy(char* szDest, const char* szSource, uint32_t ulLen)
{
  return strncpy(szDest, szSource, ulLen);
}

#ifdef CIFX_TOOLKIT_ENABLE_DSR_LOCK
/*****************************************************************************/
/*! This functions needs to provide a lock against the interrupt service 
*   routine of the device. The easiest way is an IRQ lock but some operating 
*   systems provide a way to lock against a specific interrupt
*     \param pvODDependent OS Dependent parameter in DEVICEINSTANCE          */
/*****************************************************************************/
void OS_IrqLock(void* pvOSDependent)
{
  UNREFERENCED_PARAMETER(pvOSDependent);
  EnableInterrupts;   
}

/*****************************************************************************/
/*! This function re-enables the device's interrupt service routine.
*     \param pvODDependent OS Dependent parameter in DEVICEINSTANCE          */
/*****************************************************************************/
void OS_IrqUnlock(void* pvOSDependent)
{
  UNREFERENCED_PARAMETER(pvOSDependent);
  DisableInterrupts;
}

#endif

